#!/bin/bash

PATCH_NO="t2358"
OS_VERSIONS=(8)
VERSIONS=("^ent-4.3.1")
NG_VERSIONS=()

# Do not edit below
LOG="./patch.log"
VERSION_FILE='/opt/passlogic/VERSION'
CURRENT_VERSION=`head -1 $VERSION_FILE`
OS_VERSION=`/bin/sed 's/.*release\s\([0-9]\).*/\1/' /etc/redhat-release`
PATCHED_VERSION="${CURRENT_VERSION}_${PATCH_NO}"
FILE_LIST="./file/file_list_${OS_VERSION}"
PATCH_DIR="./file/patch_files_${OS_VERSION}"

cmd() { date >> $LOG; echo $* | tee -a $LOG; $* 2>&1 | tee -a $LOG; return ${PIPESTATUS[0]}; }

check_version() {
	# 適用済 => 1
	grep -q -E "${PATCH_NO}$|_${PATCH_NO}_" <<< $CURRENT_VERSION && return 1

	for v in ${NG_VERSIONS[@]}
	do
		# パッチ適用不可バージョン => 1
		grep -q -E $v <<< $CURRENT_VERSION && return 1
	done

	for v in ${VERSIONS[@]}
	do
		# パッチ適用可能バージョン => 0
		grep -q -E $v <<< $CURRENT_VERSION && return 0
	done

	# 適用対象バージョンにマッチしない => 1
	return 1
}

update_version_file() { echo ${PATCHED_VERSION} > $VERSION_FILE; }

check_os_version() {
	for v in ${OS_VERSIONS[@]}
	do
		# OS バージョン適合 => 0
		grep -q -E "^$v$" <<< $OS_VERSION && return 0
	done

	# OS バージョン非適合 => 1
	return 1
}

backup_files() {
	# ファイル一覧をログ出力
	echo "-- File list begin --" >> $LOG
	cat $FILE_LIST >> $LOG
	echo "-- File list end --" >> $LOG
	cmd rsync -a -r --files-from=$FILE_LIST / ./backup_files/
}

revert_files() {
	# ファイル一覧をログ出力
	echo "-- File list begin --" >> $LOG
	cat $FILE_LIST >> $LOG
	echo "-- File list end --" >> $LOG
	cmd rsync -a -r --files-from=$FILE_LIST ./backup_files/ /
}

check_backup_files() {
	catln $FILE_LIST | while read file
	do
		grep -q -E "^#|^\s*$|^\s+" <<< $file && continue
		[ -r ./backup_files/$file ] || return 1
	done
}

dialog_update() {
	dialog "Start the updating process for ${PATCHED_VERSION}? [yes/no]"
}

dialog_revert() {
	dialog "Start reverting the patched files? [yes/no]"
}

dialog() {
	local agreed
	while [ -z "$agreed" ]; do
		echo
		echo $1
		read reply leftover
		case $reply in
			[yY] | [yY][eE][sS])
				agreed=1
				;;
			[nN] | [nN][oO])
				return 1
				;;
		esac
	done
	return 0
}

catln() {
	echo "`cat $@`"
}

die() {
	echo $* | tee -a $LOG
	exit 1
}

ch_own_mod() {
	local owner=$1
	local mode=$2
	local dst=$3

	( [ $owner = '-' ] || cmd chown -R $owner $dst ) && \
		( [ $mode = '-' ] || cmd chmod -R $mode $dst ) || return 1
}


deploy_files() {
	local owner=root
	local group=root
	local mode=644
	local strip_mode_line

	catln $FILE_LIST | while read file
	do
		# (1) ファイル所有者：グループ：モード行
		# owner group mode を設定、次の行に進む
		grep -q -E "^#\s+(\S+):(\S+):(\S+)\s*$" <<< $file && \
			strip_mode_line=`sed -n "s/^#\s\+\(\S\+\)\s*$/\1/p" <<< "${file}"` && \
			IFS=':' read owner group mode <<< "${strip_mode_line}" && \
			continue

		# (2) 以下に適合する行はスキップ
		#   o /opt/passlogic/VERSION
		#   o ## で始まる行
		#   o 空行、空白のみの行
		grep -q -E "^/opt/passlogic/VERSION$|^##|^\s*$|^\s+" <<< $file && \
			continue

		# (3) 以下にマッチしたらループを抜ける
		#   # End of list for patch files
		#   これ以下の行も rsync では参照する
		grep -q -E "^#\s+End of list for patch files\s*$" <<< $file && \
			break
		
		# (4) それ以外の行は、パッチ対象ファイルとして、配置
		cmd cp -af $PATCH_DIR$file $file && \
			cmd chown ${owner}:${group} $file && \
			cmd chmod $mode $file || return 1
	done
	[ $? -eq 0 ] && return 0 || return 1
}

exec_patch() {
	deploy_files && \
		update_version_file
	return $?
}


### Main-------------------------

case "$1" in
	exec)
		TENANT='single'
		check_version && check_os_version || die "Invalid version of PassLogic: $CURRENT_VERSION OS: RHEL${OS_VERSION}"
		dialog_update || die "Quit. No files patched."
		backup_files || die "Failed to back up the files. No files patched."
		exec_patch
		;;
	undo)
		TENANT='single'
		dialog_revert || die "Quit. No files reverted."
		check_backup_files || die "Backup files error - No files reverted."
		revert_files || die "Failed to revert the files."
		;;
	*)
		echo "Usage : $1 {exec|undo}"
		exit 1
		;;
esac

if [ $? -eq 0 ]
then
	echo Terminated successfully.
else
	echo Error: Aborted.
fi
