#!/bin/bash

# - 各指定ファイル
#   - ファイルを標準出力
#   - 改行を追加
#
catln() {
	local ret=true
	for f in "$@"
	do
		cat $f
		if [ $? -ne 0 ]
		then
			ret=false
		fi
		echo -e "\n"
	done
	$ret || return 1
}

cmd() {
	local array=$1 ret
	if [[ $1 == "-stderr" ]]
	then
		local temp=$(mktemp -t installer.XXXXXXXX)
		log INFO: "$@"
		"${@:2}" 2> $temp | tee -a $LOG_FILE
		ret=${PIPESTATUS[0]}
		if [ $ret -ne 0 ]
		then
			cat $temp >&2
			sed '$a\' $temp >> $LOG_FILE
			rm -f $temp
			exit $ret
		fi
	else
		log INFO: "$@"
		"$@" 2>&1 | tee -a $LOG_FILE
		ret=${PIPESTATUS[0]}
		[ $ret -eq 0 ] || exit $ret
	fi
}

dialog() {
	local agreed
	while [ -z "$agreed" ]; do
		echo
		echo $1
		read reply leftover
		case $reply in
			[yY] | [yY][eE][sS])
				agreed=1
				;;
			[nN] | [nN][oO])
				return 1
				;;
		esac
	done
	return 0
}

die() {
	echo -e "$@"
	log "$@"
	exit 1
}

log() {
    echo -e `date "+%Y-%m-%d %H:%M:%S"` "$@" >> $LOG_FILE
}

msg() {
    echo -e `date "+%Y-%m-%d %H:%M:%S"` "$@"
    echo -e `date "+%Y-%m-%d %H:%M:%S"` "$@" >> $LOG_FILE
}


parse_version() {
  local version_string=$1 version_array=() version

  # Validation
  if ! err=$(grep -qE '^ent-[0-9]{1,}\.[0-9]{1,}\.[0-9]{1,}$' <<< "${version_string}" 2>&1)
  then
      echo 'Invalid version' "${err}"
      return 1
  fi

  # ent- をカット
  if ! version_string=$(cut -c 5- <<< "${version_string}" 2>&1)
  then
      echo 'unknown error' "${version_string}"
  fi
  
  IFS='.' read -ra version <<< "$version_string"

  for v in "${version[@]}"; do
    version_array+=("$v")
  done

  echo "${version_array[@]}"
}

revision() {
	local version_string=$1

	if ! version_array=($(parse_version $1))
	then
		echo $version_array
		return 1
	fi

	echo "${version_array[-1]}"
}

# Setup global variables
# - VERSION (現在のバージョン)
# - TENANT (テナントモード; single / multi)
# - LOG_FILE
# - BACKUP_DIR ()
# - INSTALLER_VERSION (インストーラのバージョン)
#
init_global_variables() {
	local version_file='/opt/passlogic/VERSION'
	local multitenant_setting_php='/opt/passlogic/apps/lib/settings/multitenant_setting.php'

	if [[ ! -e $version_file ]]
	then
		echo "${version_file} not found"
		exit 1
	fi
	VERSION=$(head -1 "${version_file}" | sed 's/[\n\r]//g')

	if [[ ! -e $multitenant_setting_php ]]
	then
		echo "${multitenant_setting_php} not found"
		exit 1
	fi
	php -r "require_once('${multitenant_setting_php}');exit(MULTI_TENANT);"
	if [ $? -eq 0 ]
	then
		TENANT='single'
	else
		TENANT='multi'
	fi

	# LOG_FILE
	LOG_FILE=$(dirname $SCRIPT)/install.log

	# BACKUP_DIR
	BACKUP_DIR=$(dirname $SCRIPT)/backup

	# INSTALLER_VERSION
	INSTALLER_VERSION="passlogic-${BRANCH}.${REVISIONS[-1]}"
}

check_os_version() {
	local redhat_release='/etc/redhat-release'

	if [[ ! -e $redhat_release ]]
	then
		die "サポート対象外のオペレーティングシステムです。 - Unsupported operating system."
	fi
	
	local os_version=$(sed 's/.*release\s\([0-9]\).*/\1/' $redhat_release)
	
	if ! [[ ${OS:2} = $os_version ]]
	then
		die "インストーラーが、このOSのバージョンと互換性がないため、インストールできません。 -  The installer is incompatible with the version of the operating system, and cannot be installed."
	fi
}

check_passlogic_version() {
	local current_version=$1 branch=$2 installer_version=$3 patch_revisions=("${@:4}")
	local current_version_array new_version new_version_array

	new_version="${branch}.${patch_revisions[-1]}"

	if ! new_version_array=($(parse_version $new_version))
	then
		die $new_version_array
	fi

	if ! current_version_array=($(parse_version $current_version))
	then
		die $current_version_array
	fi

	if (( current_version_array[0] != new_version_array[0] || current_version_array[1] != new_version_array[1] ))
	then
		die "現在のバージョン (${current_version}) とインストーラのバージョン (${installer_version}) が一致していないため、インストールできません。 - The installation cannot proceed because the current version (${current_version}) and the installer version (${installer_version}) do not match."
	fi

	if (( current_version_array[2] >= new_version_array[2] ))
	then
		die "現在のリビジョン (${current_version}) は、このインストーラのリビジョン (${installer_version}) と等しいか新しい為、インストールできません。 - Current revision, ${current_version}, is equal to or newer than this installer ($installer_version) ."
	fi
}

check_patches() {
	local patch_dir=$1 version=$2 branch=$3 revisions=("${@:4}")
	local next_rev_index flag=true

	if ! next_rev_index=$(revision $version)
	then
		die $next_rev_index
	fi
	
	for rev in "${revisions[@]}"
	do
		local patch="${patch_dir}/${branch}.$rev"
		if ! [[ -d "$patch" ]]
		then
			echo "$patch missing"
			flag=false
		fi
	done
	$flag || die "Resource files missing"
}

check_backup() {
	local current_version=$1 branch=$2 installer_version=$3 backup_dir=$4
	local current_version_array old_version old_version_array old_file_version

	set -o noglob
	if ! old_version=($(find $backup_dir -type d -name "${branch}.*" -printf '%f\n' 2>&1 ))
	then
		die "バックアップが見つかりません。 - Backup not found.;  ${old_version[@]}"
	fi

	if (( ${#old_version[@]} != 1 ))
	then
		die "不正なバックアップです。 - Invalid backup."
	fi

	if ! old_version_array=($(parse_version ${old_version[0]}))
	then
		die $old_version_array
	fi

	if ! old_version_in_file=$(cat ${backup_dir}/${old_version[0]}/opt/passlogic/VERSION 2>&1)
	then
		die "バックアップに VERSION ファイルがありません。 - Missing VERSION file in backup.  $old_version_in_file"
	fi

	if [[ "${old_version[0]}" != "${old_version_in_file}" ]]
	then
		die "不正なバックアップ/バージョンファイルです。 - Invalid backup/version file."
	fi		
	
	if [[ ! -e "${backup_dir}/file_list" ]]
	then
		die "バックアップに file_list がありません。 - 'file_list' is missing from the backup."
	fi

	if ! current_version_array=($(parse_version $current_version))
	then
		die $current_version_array
	fi

	if (( current_version_array[0] != old_version_array[0] || current_version_array[1] != old_version_array[1] ))
	then
		die "現在のバージョン (${current_version}) とバックアップのバージョン (${old_version}) が一致していないため、切り戻しできません。 - The backup cannot be restored because the current version (${current_version}) does not match the backup version (${old_version})."
	fi

	if (( current_version_array[2] <= old_version_array[2] ))
	then
		die "現在のリビジョン (${current_version}) は、バックアップのリビジョン (${old_version}) と同じか、それよりも古いバージョンになっているため、切り戻しできません。 - The current revision (${current_version}) is equal to or older than the backup revision (${old_version}), so the rollback cannot be performed."
	fi

	echo "${old_version}"
}

product_of() {
	local status pipestatus=("$@") product=true
	
	for status in "${pipestatus[@]}"
	do
		if ! [[ $status = 0 ]]
		then
			product=false
		fi
	done

	$product || return 1
}

files_test() {
	local array=("$@") file files_ok=true errors=""

	for file in "${array[@]}"
	do
		if [ ! -e "$file" ]
		then
			files_ok=false
			errors+="$file not exist\n"
		elif [ ! -r "$file" ]
		then
			files_ok=false
			errors+="$file not readable\n"
		fi
	done

	if $files_ok
	then
		return 0
	else
		echo -e "$errors"
		return 1
	fi
}

make_backup() {
	local patch_dir=$1 backup_dir=$2 os_version=$3 version=$4 branch=$5 file_list_merged=$6 revisions=("${@:7}")
	local ts=$(date +%Y%m%d%H%M%S) res

	if [[ -d $backup_dir ]]
	then
		log INFO: "Rename old backup directory ${backup_dir} to ${backup_dir}_$ts"
		if ! res=$(mv "${backup_dir}" "${backup_dir}_old_$ts" 2>&1)
		then
			die "$res"
		fi
	fi

	local version_array
	if ! version_array=($(parse_version $version))
	then
		die $version_array
	fi
	
	local dir="${backup_dir}/${version}"
	cmd mkdir -p $dir

	local rev file_list=()	new_file_list=()
	for rev in "${revisions[@]:${version_array[-1]}}"
	do
		file_list+=( "${patch_dir}/${branch}.${rev}/file/file_list_${os_version}" )
		new_file_list+=("${patch_dir}/${branch}.${rev}/file/new_file_list_${os_version}" )
	done

	if ! res=$(make_files_from "${backup_dir}/${file_list_merged}.tmp" "${file_list[@]}")
	then
		die "$res"
	fi

	if ! res=$(make_exclude_list "${backup_dir}/${file_list_merged}.exclude" "${new_file_list[@]}")
	then
		die "$res"
	fi

	cmd -stderr grep -vFf "${backup_dir}/${file_list_merged}.exclude" "${backup_dir}/${file_list_merged}.tmp" > "${backup_dir}/${file_list_merged}"

	cmd rsync -a  -r --files-from="${backup_dir}/${file_list_merged}" / "${backup_dir}/${version}"

}

restore_backup() {
	local backup_dir=$1 file_list=$2 version=$3

	cmd rsync -a -r --files-from="${backup_dir}/${file_list}" "${backup_dir}/${version}" /

}

make_files_from() {
	local file_list_merged=$1 file_lists=("${@:2}")
	local res

	if ! res=$(files_test "${file_lists[@]}")
	then
		echo -e "$res"
		return 1
	fi

	if ! res=$((catln "${file_lists[@]}" | sort | uniq | grep -v -E '^#|^\s*$' > ${file_list_merged}; return $(product_of "${PIPESTATUS[@]}")) 2>&1 )
	then
		rm -f ${file_list_merged}
		echo -e "$res"
		return 1
	fi
}

make_exclude_list() {
	local exclude_list=$1 new_file_list=("${@:2}")
	local  flag res

	cmd rm -f "${exclude_list}"
	if ! res=$(files_test "${new_file_list[@]}")
	then
		die "$res"
	fi

	touch $exclude_list
	catln "${new_file_list[@]}" | while read line
	do
		[ -z "$line" ] && continue

		grep -q -E '^#(file|symlink|touch|ch_own_mod|directory)\s*$' <<< $line && \
			flag=`sed 's/^#\(\S\+\)\s*$/\1/' <<< $line` && \
			continue

		echo $line
		
		grep -q -E '^\S+(,\S+){4}$' <<< $line && \
			IFS=',' read src dst owner mode tenant <<< "$line" && \
			[ $tenant = $TENANT -o $tenant = '-' ] && case $flag in
				file|symlink|touch|directory)
					echo $dst >> ${exclude_list}
					;;
				*)
					;;
			esac
	done
}

update_version_file() {
	local version_file='/opt/passlogic/VERSION' version=$1
	cmd echo "$version" > "$version_file"
}

SCRIPT=$(readlink  -f $0)
PATCH_DIR=$(dirname $SCRIPT)
INIT_FILE=${SCRIPT/.sh/.init}
. $INIT_FILE
init_global_variables
check_os_version

case $1 in
	update)
		check_passlogic_version "$VERSION" "$BRANCH" "${INSTALLER_VERSION}" "${REVISIONS[@]}"
		check_patches "${PATCH_DIR}" "${VERSION}" "$BRANCH" "${REVISIONS[@]}"

		dialog "${BRANCH}.${REVISIONS[-1]} へのアップデートを開始しますか？ - Would you like to start the update? [yes/no] "
		make_backup "${PATCH_DIR}" "${BACKUP_DIR}" "${OS:2}" "${VERSION}" "$BRANCH" "file_list" "${REVISIONS[@]}"
		for rev in "${REVISIONS[@]:$(revision $VERSION)}"
		do
			echo "${PATCH_DIR}/${BRANCH}.${rev}/patch.sh" patch
			"${PATCH_DIR}/${BRANCH}.${rev}/patch.sh" patch
			if [ $? -ne 0 ]
			then
				echo error
				exit 1
			fi
			echo ""
		done

		update_version_file "${BRANCH}.${REVISIONS[-1]}"
		
		msg INFO: "アップデートが完了しました。現在のバージョンは ${BRANCH}.${REVISIONS[-1]} です。" \
			 "- The update has been completed. The current version is ${BRANCH}.${REVISIONS[-1]}."
		
		;;
	revert)
		if ! backup_version=$(check_backup "$VERSION" "$BRANCH" "${INSTALLER_VERSION}" "${BACKUP_DIR}")
		then
			die $backup_version
		fi

		dialog "$backup_version に切り戻しますか？ - Would you like to roll back ? [yes/no] "

		restore_backup "${BACKUP_DIR}" "file_list" "$backup_version"
		old_rev=$(revision $backup_version)
		revisions=("${REVISIONS[@]:${old_rev}}")
		for (( i=${#revisions[@]}-1; i>=0; i-- ))
		do
			"${PATCH_DIR}/${BRANCH}.${revisions[$i]}/patch.sh" revert
		done

		msg INFO: " 切り戻しが完了しました。現在のバージョンは $(cat /opt/passlogic/VERSION) です。" \
			 "- The revert has been completed. The current version is $(cat /opt/passlogic/VERSION) ."
		
		;;
	*)
		echo "./install.sh [update|revert]"
		exit 1
		;;
esac
