#!/bin/bash

# Do not edit below
LOG="./install.log"
VERSION_FILE='/opt/passlogic/VERSION'
CURRENT_VERSION=`head -1 $VERSION_FILE`
OS_VERSION=`/bin/sed 's/.*release\s\([0-9]\).*/\1/' /etc/redhat-release`
SCRIPT_DIR=$(dirname $(readlink -f $0))
FILE_LIST="${SCRIPT_DIR}/file/file_list_${OS_VERSION}"
NEW_FILE_LIST="${SCRIPT_DIR}/file/new_file_list_${OS_VERSION}"
PATCH_DIR="${SCRIPT_DIR}/file/patch_files_${OS_VERSION}"
PM=/opt/passlogic/apps/tools/pm.sh

php -r 'require_once("/opt/passlogic/apps/lib/settings/multitenant_setting.php");exit(MULTI_TENANT);'
if [ $? -eq 0 ]
then
	TENANT='single'
else
	TENANT='multi'
fi


cmd() { date >> $LOG; echo $* | tee -a $LOG; $* 2>&1 | tee -a $LOG; return ${PIPESTATUS[0]}; }

product_of() {
	local status pipestatus=("$@") product=true
	
	for status in "${pipestatus[@]}"
	do
		if ! [[ $status = 0 ]]
		then
			product=false
		fi
	done

	$product || return 1
}

check_version() {
	# 適用済 => 1
	grep -q -E "${PATCH_NO}$|_${PATCH_NO}_" <<< $CURRENT_VERSION && return 1

	for v in ${NG_VERSIONS[@]}
	do
		# パッチ適用不可バージョン => 1
		grep -q -E $v <<< $CURRENT_VERSION && return 1
	done

	for v in ${VERSIONS[@]}
	do
		# パッチ適用可能バージョン => 0
		grep -q -E $v <<< $CURRENT_VERSION && return 0
	done

	# 適用対象バージョンにマッチしない => 1
	return 1
}

check_os_version() {
	for v in ${OS_VERSIONS[@]}
	do
		# OS バージョン適合 => 0
		grep -q -E "^$v$" <<< $OS_VERSION && return 0
	done

	# OS バージョン非適合 => 1
	return 1
}

backup_files() {
	# ファイル一覧をログ出力
	echo "-- File list begin --" >> $LOG
	cat $FILE_LIST >> $LOG
	echo "-- File list end --" >> $LOG
	cmd rsync -a -r --files-from=$FILE_LIST / ./backup_files/
}

revert_files() {
	# ファイル一覧をログ出力
	echo "-- File list begin --" >> $LOG
	cat $FILE_LIST >> $LOG
	echo "-- File list end --" >> $LOG
	cmd rsync -a -r --files-from=$FILE_LIST ./backup_files/ /
}

check_backup_files() {
	catln $FILE_LIST | while read file
	do
		grep -q -E "^#|^\s*$|^\s+" <<< $file && continue
		[ -r ./backup_files/$file ] || return 1
	done
}

dialog_update() {
	dialog "Start the updating process for ${PATCHED_VERSION}? [yes/no]"
}

catln() {
	echo "`cat $@`"
}

die() {
	echo $* | tee -a $LOG
	exit 1
}

ch_own_mod() {
	local owner=$1
	local mode=$2
	local dst=$3

	( [ $owner = '-' ] || cmd chown -R $owner $dst ) && \
		( [ $mode = '-' ] || cmd chmod -R $mode $dst ) || return 1
}

install_new_files() {
	local src dst owner mode tenant
	local flag ret=true

	catln $NEW_FILE_LIST | while read line
	do
		[ -z "$line" ] && continue

		grep -q -E '^#(file|symlink|touch|ch_own_mod|directory)\s*$' <<< $line && \
			flag=`sed 's/^#\(\S\+\)\s*$/\1/' <<< $line` && \
			continue

		grep -q -E '^\S+(,\S+){4}$' <<< $line && \
			IFS=',' read src dst owner mode tenant <<< "$line" && \
			[ $tenant = $TENANT -o $tenant = '-' ] && case $flag in
				file)
					cmd cp -af $PATCH_DIR$src $dst && ch_own_mod $owner $mode $dst || return 1
					;;
				symlink)
					cmd ln -s $src $dst || return 1
					;;
				touch)
					cmd touch $dst && \
						ch_own_mod $owner $mode $dst || return 1
					;;
				ch_own_mod)
					ch_own_mod $owner $mode $dst || return 1
					;;
				directory)
					cmd mkdir -p $dst && \
					ch_own_mod $owner $mode $dst || return 1
					;;
				*)
					return 1
					;;
			esac
	done

	return $(product_of "${PIPESTATUS[@]}")
}

uninstall_new_files() {
	local src dst owner mode tenant
	local flag

	catln $NEW_FILE_LIST | while read line
	do
		[ -z "$line" ] && continue

		grep -q -E '^#(file|symlink|touch|ch_own_mod|directory)\s*$' <<< $line && \
			flag=`sed 's/^#\(\S\+\)\s*$/\1/' <<< $line` && \
			continue

		grep -q -E '^\S+(,\S+){4}$' <<< $line && \
			IFS=',' read src dst owner mode tenant <<< "$line" && \
			case $flag in
				file)
					cmd rm -rf $dst
					;;
				symlink)
					cmd rm -f $dst
					;;
				touch)
					cmd rm -f $dst
					;;
				directory)
					cmd rm -rf $dst
					;;
				*)
					;;
			esac
	done
	return $(product_of "${PIPESTATUS[@]}")
}

deploy_files() {
	local owner=root
	local group=root
	local mode=644
	local strip_mode_line

	catln $FILE_LIST | while read file
	do
		# (1) ファイル所有者：グループ：モード行
		# owner group mode を設定、次の行に進む
		grep -q -E "^#\s+(\S+):(\S+):(\S+)\s*$" <<< $file && \
			strip_mode_line=`sed -n "s/^#\s\+\(\S\+\)\s*$/\1/p" <<< "${file}"` && \
			IFS=':' read owner group mode <<< "${strip_mode_line}" && \
			continue

		# (2) 以下に適合する行はスキップ
		#   o /opt/passlogic/VERSION
		#   o /opt/passlogic/apps/lib/settings/global_setting.php
		#   o ## で始まる行
		#   o 空行、空白のみの行
		grep -q -E "^/opt/passlogic/VERSION$|^/opt/passlogic/apps/lib/settings/global_setting.php$|^##|^\s*$|^\s+" <<< $file && \
			continue

		# (3) 以下にマッチしたらループを抜ける
		#   # End of list for patch files
		#   これ以下の行も rsync では参照する
		grep -q -E "^#\s+End of list for patch files\s*$" <<< $file && \
			break
		
		# (4) それ以外の行は、パッチ対象ファイルとして、配置
		cmd cp -af $PATCH_DIR$file $file && \
			cmd chown ${owner}:${group} $file && \
			cmd chmod $mode $file || return 1
	done

	return $(product_of "${PIPESTATUS[@]}")
}

mod_global_setting_php() {
	local conf=/opt/passlogic/apps/lib/settings/global_setting.php
	local tmp_php=/var/tmp/global_setting.php
	local php_tag_closed=true

	\rm -f $tmp_php
	grep -v \
		 -e 'multitenant_setting.php' \
		 -e 'session.inc.php' \
		 -e '/validator.inc.php' \
		 -e '/form.inc.php' \
		 -e '/util.inc.php' \
		 -e '/crypt.inc.php' \
		 -e '/database/passlogic_config.php' \
		 -e '/model/admin_model.php' \
		 -e '/model/user_model.php' \
		 -e '/tenant_controller.php' \
		 -e '/passlogic.inc.php' \
		 -e '/passlogic_api_util.inc.php' \
		 -e '/log.inc.php' \
		 -e '/cert_util.inc.php' \
		 $conf > $tmp_php || return 1

	# 改行で終わっていなければ改行を追加
	local last_char=$(tail -c 1 $conf) || return 1
	if [ "$last_char" != "" ]
	then
		echo >> $conf || return 1
		echo >> $tmp_php || return 1
	fi
	
	cat <<EOF >> $tmp_php
<?php
echo 'php closing tag test';
EOF

	php $tmp_php > /dev/null 2>&1

	if [ $? -ne 0 ]
	then
		php_tag_closed=false
	fi

	if $php_tag_closed
	then
		echo '<?php' >> $conf || return 1
	fi

	# global_setting.phpへ追記する内容
	cat <<EOF >> $conf || return 1
EOF
}

remove_passlogic_lang_cache() {
	local cache_file=/opt/passlogic/tmp/passlogic-lang.cache
	\rm -f $cache_file
}

remove_passlogic_log_cache() {
	local cache_file=/opt/passlogic/tmp/passlogic-log.cache
	\rm -f $cache_file
}

exec_patch() {
	deploy_files
	return $?
}

### Main-------------------------

case "$1" in
	patch)
		install_new_files || die "Failed to install the new files."
		exec_patch || die "Failed to patch files."
#		mod_global_setting_php || die "Failed to update 'global_setting.php' ."
#		remove_passlogic_lang_cache
#		remove_passlogic_log_cache
		;;
	revert)
		uninstall_new_files || die "Failed to remove the files."
		;;
	*)
		echo "Usage : $1 {patch|revert}"
		exit 1
		;;
esac

exit $?
